<?php
/**

 * NOTICE OF LICENSE

 *

 * This file is licenced under the Software License Agreement.

 * With the purchase or the installation of the software in your application

 * you accept the licence agreement.

 *

 * You must not modify, adapt or create derivative works of this source code

 *

 *  @author    Coccinet

 *  @copyright 2017 Coccinet

 *  @license   LICENSE.txt

 */

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

if (!defined('_PS_VERSION_')) {
    exit;
}
require_once(dirname(__FILE__).'/classes/common.inc.php');

class ElavonPaymentsHpp extends PaymentModule
{
    protected $_html = '';
    protected $_postErrors = array();
    protected $_config;

    public function __construct()
    {
        $this->name = 'Elavonpaymentshpp';
        $this->tab = 'payments_gateways';
        $this->version = '2.0.0';
        $this->author = 'Coccinet';
        $this->controllers = array('payment', 'validation');
        $this->is_eu_compatible = 0;
        $this->_config = new ElavonPaymentsConfig();

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = $this->l('Elavon Payments HPP');
        $this->description = $this->l('Use Elavon Payments as your payments service provider.');
        $this->confirmUninstall = $this->l('Are you sure about removing these details?');
        $this->ps_versions_compliancy = array('min' => '1.7', 'max' => '1.7.99.99');

        if (!function_exists('curl_version')) {
            $this->warning = $this->l('cURL librairy is not available.');
        } elseif (!Configuration::get('PS_REWRITING_SETTINGS')) {
            $this->warning = $this->l('URL Rewriting must be enabled before using this module.');
        } elseif (!isset($this->merchant_id)
                || empty($this->shared_secret)
                || empty($this->subaccount)) {
            $this->warning = $this->l('Elavon Payments details must be configured before using this module.');
        }
    }

    public function install()
    {
        if (!parent::install()) {
                return false;
        }

        require_once(dirname(__FILE__) . '/classes/ElavonPaymentsInstaller.php');
        $installer = new ElavonPaymentsInstaller();
        $hooks = array('hookPaymentOptions', 'paymentReturn', 'header','displayShoppingCart','DisplayAdminOrderLeft','actionObjectLanguageAddAfter','paymentOptions');
        if (!$installer->installModule($this, $hooks) || !$this->cleanLogs()) {
                return false;
        }
        return true;
    }

    public function uninstall()
    {
        if (!Configuration::deleteByName('ELAVONPAYMENTS_PRODUCTION')
            || !Configuration::deleteByName('ELAVONPAYMENTS_ENABLE_LOGS')
            || !Configuration::deleteByName('ELAVONPAYMENTS_MERCHANT_ID')
            || !Configuration::deleteByName('ELAVONPAYMENTS_SHARED_SECRET')
            || !Configuration::deleteByName('ELAVONPAYMENTS_REBATE_PASSWORD')
            || !Configuration::deleteByName('ELAVONPAYMENTS_SUBACCOUNT')
            || !Configuration::deleteByName('ELAVONPAYMENTS_URL_LIVE')
            || !Configuration::deleteByName('ELAVONPAYMENTS_URL_SANDBOX')
            || !Configuration::deleteByName('ELAVONPAYMENTS_PAYMENT_TEXT')
            || !Configuration::deleteByName('ELAVONPAYMENTS_HPP_LANG')
            || !Configuration::deleteByName('ELAVONPAYMENTS_HPP_LANG_ISO')
            || !Configuration::deleteByName('ELAVONPAYMENTS_AUTO_SETTLE')
            || !Configuration::deleteByName('ELAVONPAYMENTS_IFRAME')
            || !Configuration::deleteByName('ELAVONPAYMENTS_IFRAME_TYPE')
            || !Configuration::deleteByName('ELAVONPAYMENTS_CARD_STORAGE')
            || !Configuration::deleteByName('ELAVONPAYMENTS_OFFER_SAVE_CARD')
            || !$this->cleanLogs()
            || !parent::uninstall()) {
            return false;
        }
        return true;
    }

    public function getConfig()
    {
        return $this->_config;
    }

    protected function _postValidation()
    {
        if (Tools::isSubmit('btnSubmitPrin')) {
            if (!Configuration::get('PS_REWRITING_SETTINGS')) {
                $this->_postErrors[] = $this->l('URL Rewriting must be enabled before using this module.');
            }
            if (!function_exists('curl_version')) {
                $this->_postErrors[] = $this->l('cURL librairy is not available.');
            }
            if (!Tools::getValue('ELAVONPAYMENTS_MERCHANT_ID')) {
                $this->_postErrors[] = $this->l('Merchant ID is required.');
            }
            if (!Tools::getValue('ELAVONPAYMENTS_SUBACCOUNT')) {
                $this->_postErrors[] = $this->l('Subaccount is required');
            }
            if (!Tools::getValue('ELAVONPAYMENTS_URL_LIVE')) {
                $this->_postErrors[] = $this->l('Live URL is required');
            }
            if (!Tools::getValue('ELAVONPAYMENTS_URL_SANDBOX')) {
                $this->_postErrors[] = $this->l('Sandbox URL is required');
            }
            if (!Configuration::get('PS_REWRITING_SETTINGS')) {
                $this->_postErrors[] = $this->l('URL Rewriting must be enabled before using this module.');
            }
            if (!function_exists('curl_version')) {
                $this->_postErrors[] = $this->l('cURL librairy is not available.');
            }
            if (Tools::getValue('ELAVONPAYMENTS_HPP_LANG') !='customer' && !Tools::getValue('ELAVONPAYMENTS_HPP_LANG_ISO')) {
                $this->_postErrors[] = $this->l('HPP language is required');
            }
            $languages = Language::getLanguages(false);
            foreach ($languages as $lang) {
                if (!Tools::getValue('ELAVONPAYMENTS_PAYMENT_TEXT_'.$lang['id_lang'])) {
                    $this->_postErrors[] = $this->l('Payment text is required for '.$lang['iso_code']);
                }
            }
        }
        if (Tools::isSubmit('btnSubmitSec')) {
            if (!Tools::getValue('ELAVONPAYMENTS_SHARED_SECRET')) {
                $this->_postErrors[] = $this->l('Shared secret is required.');
            }
            if (!Tools::getValue('ELAVONPAYMENTS_REBATE_PASSWORD')) {
                $this->_postErrors[] = $this->l('Rebate Password is required.');
            }
        }
    }

    public function getContent()
    {
        if (Tools::getValue('actionClear')) {
            $this->cleanFails();
        }
        require_once(dirname(__FILE__) . '/classes/ElavonPaymentsAdminConfig.php');


        $admin = new ElavonPaymentsAdminConfig($this);
        if (Tools::isSubmit('btnSubmitPrin') || Tools::isSubmit('btnSubmitSec') || Tools::isSubmit('btnClearRecords')) {
            $this->_postValidation();
            if (!count($this->_postErrors)) {
                $admin->_postProcess();
            } else {
                foreach ($this->_postErrors as $err) {
                    $this->_html .= $this->displayError($err);
                }
            }
        } else {
            $this->_html .= '<br />';
        }
        $this->_html .= $admin->getContent($this->_config);
        return $this->_html;
    }
    public function hookPaymentOptions($params)
    {
        if (!$this->active) {
            return;
        }

        if (!$this->checkCurrency($params['cart'])) {
            return;
        }

        $payment_options = array(
            $this->getElavonPaymentOption(),
            );

        return $payment_options;
    }
    public function hookActionObjectLanguageAddAfter($params)
    {
        $values = array();
        $values['ELAVONPAYMENTS_PAYMENT_TEXT'][(int)$params['object']->id] = 'Pay by Credit or Debit Card';
        return Configuration::updateValue('ELAVONPAYMENTS_PAYMENT_TEXT', $values['ELAVONPAYMENTS_PAYMENT_TEXT']);
    }

    public function getElavonPaymentOption()
    {
        //get the payment text defined on backoffice
        $payment_text_array = $this->_config->getPaymentText();
        //set default text if it's empty
        if (!$payment_text_array[$this->context->language->id]) {
            $payment_text = "Pay By Credit or Debit Card";
        } else {
            $payment_text = $payment_text_array[$this->context->language->id];
        }
        $ElavonPaymentOption = new PaymentOption();
        $ElavonPaymentOption->setCallToActionText($payment_text)
                      ->setAction($this->context->link->getModuleLink($this->name, 'payment', array(), true))
                    /*->setAdditionalInformation($this->context->smarty->fetch('module:Elavonpaymentshpp/views/templates/front/payment_info.tpl'))*/;

         return $ElavonPaymentOption;
    }
    public function hookPaymentReturn($params)
    {
        if (!$this->active) {
            return;
        }
        // var_dump($params);die;
        $state = $params['order']->getCurrentState();
        if (in_array($state, array(Configuration::get('PS_OS_PAYMENT'), Configuration::get('PS_OS_OUTOFSTOCK'), Configuration::get('ELAVONPAYMENTS_PENDING_STATUS')))) {
            $this->smarty->assign(array(
                'total_to_pay' => Tools::displayPrice($params['order']->total_paid, $params['cart']->id_currency, false),
                'status' => 'ok',
                'id_order' => $params['order']->id
            ));
            if (isset($params['order']->reference) && !empty($params['order']->reference)) {
                $this->smarty->assign('reference', $params['order']->reference);
            }
        } else {
            $this->smarty->assign('status', 'failed');
        }
        return $this->display(__FILE__, 'payment_return.tpl');
    }

    public function hookHeader($params)
    {
        $this->context->controller->addCSS($this->_path.'views/css/Elavonpayments.css', 'all');
        $this->context->controller->addJS($this->_path.'views/js/Elavonpayments.js', 'all');
        if (Tools::getValue('Elavonpayments_error')) {
            $this->smarty->assign('status', 'failed');
            return $this->display(__FILE__, 'payment_error.tpl');
        }
    }

    public function hookDisplayShoppingCart()
    {
        if (Tools::getValue('Elavonpayments_error')) {
            // $error = Tools::getValue('Elavonpayments_error');
            $this->context->controller->errors[] = $this->l('Your transaction was unsuccessful, please try again or use an alternate form of payment.');
            unset($_SERVER['HTTP_REFERER']);
            $this->context->smarty->assign('server', $_SERVER);
        }
    }

    public function hookDisplayAdminOrderLeft($params)
    {

        $this->context->controller->addCSS($this->_path.'views/css/Elavonpayments.css', 'all');
        require_once(dirname(__FILE__) . '/classes/ElavonPaymentsAdminTransactionManagement.php');
        $transaction = new ElavonPaymentsAdminTransactionManagement();
        $order = new Order($params['id_order']);
        $ElavonPaymentsOrder = new ElavonPaymentsOrders();
        $ElavonOrder = $ElavonPaymentsOrder->getOrder($order->id);
        // $obj_currency = new Currency($order->id_currency);
        // $currency = $obj_currency->sign;
        $transaction_return = "";
        $transaction_statut = "";
        $original_amount = $ElavonOrder['Elavonpayments_original_amount'];
        $request_set = false;
        // $new_state_success = false;
        if (Tools::getValue('Elavonpayments_amount')) {
            $ElavonOrder['Elavonpayments_original_amount'] = str_replace(',', '.', Tools::getValue('Elavonpayments_amount'));
        }
        if (Tools::getValue('Elavonpayments_transaction')) {
            $request_set = Tools::getValue('Elavonpayments_transaction');
            switch ($request_set) {
                case 'settle':
                    $message_success = $this->l('Settlement Successful');
                    break;
                case 'void':
                    $message_success = $this->l('Void Successful');
                    break;
                case 'rebate':
                    $message_success = $this->l('Rebate Successful');
                    break;
                default:
                    $request_set = false;
                    break;
            }
        }
        if ($request_set) {
            $request = $transaction->setRequest($ElavonOrder, $request_set);
            $xml_result = $transaction->sendRequest($request);
            $timestamp_result = $xml_result->attributes()->timestamp;
            $merchantid_result = $xml_result->merchantid;
            $orderid_result = $xml_result->orderid;
            $hash_result = $xml_result->sha1hash;
            $result_result = $xml_result->result;
            $message_result = $xml_result->message;
            $pasref_result = $xml_result->pasref;
            $authcode_result = $xml_result->authcode;
            $hash_response = ElavonPaymentsTools::getHashResponse($timestamp_result, $merchantid_result, $orderid_result, $result_result, $message_result, $pasref_result, $authcode_result, $this->getConfig()->getSharedSecret());
            $success = 0;
            switch ($result_result) {
                case '00':
                    if ($hash_response == $hash_result) {
                        $transaction_statut = "OK";
                        $transaction_return = $message_success;
                        $success = 1;
                    } else {
                        $transaction_statut = "KO";
                        $transaction_return = $this->l('Error')." : ".$this->l('Hash Error');
                    }
                    break;
                default:
                    $transaction_statut = "KO";
                    $transaction_return = $this->l('Error')." : ".$message_result;
                    break;
            }
            $amountToSave = $request_set !='void'?$ElavonOrder['Elavonpayments_original_amount']:$original_amount;
            $transaction->saveResult($order->id, $request_set, $amountToSave, $transaction_return, $success);
        }
        $this->smarty->assign('current_index', 'index.php?controller=AdminOrders');
        $this->smarty->assign('order', $order);
        $this->smarty->assign('original_amount', number_format($original_amount, 2));
        $this->smarty->assign('max_authorized', round($original_amount*1.15, 2));
        $this->smarty->assign('ElavonOrder', $ElavonOrder);
        $this->smarty->assign('transaction_return', $transaction_return);
        $this->smarty->assign('transaction_statut', $transaction_statut);
        $this->smarty->assign('transaction_history', $transaction->getHistory($order->id));

        //order message config only for 1.7
        $current_order_message = $this->_getCurrentOrderMessage($params['id_order']);
        $current_order_message_text = $current_order_message[0]["message"];
        $current_order_message_text = str_replace("\n", '<br />', $current_order_message_text);
        $current_order_message_text = str_replace("\r", '', $current_order_message_text);
        $message_date_array = date_parse($current_order_message[0]["date_add"]);
        $current_message_date = $message_date_array["month"]."/".$message_date_array["day"]."/".$message_date_array["year"];
        $message_customer = new Customer($current_order_message[0]["id_customer"]);
        $message_customer_fullname = $message_customer->firstname." ".$message_customer->lastname." ";
        $this->smarty->assign('message_text', $current_order_message_text);
        $this->smarty->assign('message_date', $current_message_date);
        $this->smarty->assign('message_customer', $message_customer_fullname);


        return $this->display(__FILE__, 'adminorder.tpl');
    }

    //get message content from database only for 1.7
    private function _getCurrentOrderMessage($id_order)
    {
        return Db::getInstance()->executeS('
            SELECT * FROM `'._DB_PREFIX_.'message`  WHERE `id_order`='.(int)$id_order);
    }

    public function checkCurrency($cart)
    {
        $currency_order = new Currency($cart->id_currency);
        $currencies_module = $this->getCurrency($cart->id_currency);
        if (is_array($currencies_module)) {
            foreach ($currencies_module as $currency_module) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
        }
        return false;
    }

    public function getResults($post)
    {
        $results = new ElavonPaymentsResults($this);
        return $results->getResults($post);
    }

    public function getLogs($results)
    {
        $logs = new ElavonPaymentsResults($this);
        return $logs->getLogs($results);
    }
    public function writeLogs($results)
    {
        $logs = new ElavonPaymentsResults($this);
        return $logs->writeLogs($results);
    }
    public function writeRequestlog($results)
    {
        $logs = new ElavonPaymentsResults($this);
        return $logs->writeRequestlog($results);
    }
    public function saveFailure($code, $message, $cart, $order_id)
    {
        $logs = new ElavonPaymentsResults($this);
        return $logs->saveFailure($code, $message, $cart, $order_id);
    }
    public function getRedirection($action, $cart = null, $customer = null, $error = null)
    {
        $redirect = new ElavonPaymentsRedirection($this);
        return $redirect->getRedirect($action, $cart, $customer, $error);
    }
    public function cleanLogs()
    {
        $file = fopen(dirname(__FILE__)."/logs/logs.txt", "w");
        fclose($file);
        return true;
    }
    public function cleanFails()
    {
        Db::getInstance()->delete('Elavonpayments_failures');
    }
}
